document.addEventListener('DOMContentLoaded', () => {
  const hamburgerButton = document.querySelector('.hamburger-button');
  const megaMenu = document.getElementById('menu-main-menu');
  const overlay = document.getElementById('menu-overlay');
  const body = document.body; // Reference to the body element
  let activeSubmenuItem = null;
  let activeSubsubmenuItem = null;

  // Function to close all submenus
  const closeAllSubmenus = () => {
    const openItems = document.querySelectorAll('.menu-item.open');
    openItems.forEach((item) => {
      item.classList.remove('open');
      const toggleButton = item.querySelector('.submenu-toggle');
      if (toggleButton) {
        toggleButton.setAttribute('aria-expanded', 'false');
      }
    });
    activeSubmenuItem = null;
    activeSubsubmenuItem = null;
    overlay.classList.remove('active');
    body.classList.remove('menu-open'); // Re-enable scrolling on the body
  };

  // Toggle hamburger menu visibility for mobile
  if (hamburgerButton && megaMenu) {
    hamburgerButton.addEventListener('click', () => {
      const isOpen = hamburgerButton.getAttribute('data-open') === 'true';
      hamburgerButton.setAttribute('data-open', isOpen ? 'false' : 'true');
      megaMenu.style.display = isOpen ? 'none' : 'block';

      if (!isOpen) {
        overlay.classList.add('active');
        body.classList.add('menu-open'); // Disable scrolling on the body
      } else {
        overlay.classList.remove('active');
        body.classList.remove('menu-open'); // Re-enable scrolling on the body
      }
    });
  }

  // Desktop behavior remains unchanged
  const setupDesktopBehavior = () => {
    const mainMenuItems = document.querySelectorAll('#menu-main-menu > .menu-item');
    const subMenuItems = document.querySelectorAll('#menu-main-menu .submenu > .menu-item');

    mainMenuItems.forEach((menuItem) => {
      const submenu = menuItem.querySelector('.submenu');

      if (submenu) {
        menuItem.querySelector('a').addEventListener('click', (e) => {
          e.preventDefault();
          const isOpen = menuItem.classList.contains('open');
          closeAllSubmenus();

          if (!isOpen) {
            menuItem.classList.add('open');
            activeSubmenuItem = menuItem;
            overlay.classList.add('active');
          }
        });
      }
    });

    subMenuItems.forEach((menuItem) => {
      const subsubmenu = menuItem.querySelector('.subsubmenu');

      if (subsubmenu) {
        menuItem.addEventListener('mouseenter', () => {
          if (activeSubsubmenuItem && activeSubsubmenuItem !== menuItem) {
            activeSubsubmenuItem.classList.remove('open');
          }
          menuItem.classList.add('open');
          activeSubsubmenuItem = menuItem;
        });

        menuItem.addEventListener('mouseleave', () => {
          if (activeSubsubmenuItem !== menuItem) {
            menuItem.classList.remove('open');
          }
        });
      }
    });
  };

  // Updated mobile behavior
  const setupMobileBehavior = () => {
    const menuItems = document.querySelectorAll('#menu-main-menu .menu-item');

    menuItems.forEach((menuItem) => {
      const submenu = menuItem.querySelector('.submenu');
      const toggleButton = menuItem.querySelector('.submenu-toggle');

      // Handle toggle button clicks
      if (toggleButton) {
        toggleButton.addEventListener('click', (e) => {
          e.stopPropagation();
          e.preventDefault();
          const isOpen = menuItem.classList.contains('open');
          menuItem.classList.toggle('open', !isOpen);
          toggleButton.setAttribute('aria-expanded', !isOpen);
        });
      }

      // Prevent submenu interaction on links
      menuItem.querySelector('a').addEventListener('click', (e) => {
        if (submenu && menuItem.classList.contains('open')) {
          // Let navigation proceed if the submenu is open
          return;
        } else if (submenu) {
          // Prevent navigation if the submenu is closed
          e.preventDefault();
        }
      });
    });

    // Ensure scrolling does not close menus
    megaMenu.addEventListener('touchmove', (e) => {
      e.stopPropagation();
    });
  };

  // Initialize behaviors based on screen size
  const updateBehavior = () => {
    if (window.matchMedia('(min-width: 1024px)').matches) {
      closeAllSubmenus();
      setupDesktopBehavior();
    } else {
      closeAllSubmenus();
      setupMobileBehavior();
    }
  };

  // Initialize and watch for screen size changes
  updateBehavior();
  window.addEventListener('resize', updateBehavior);

  // Close submenus when clicking outside
  document.addEventListener('click', (e) => {
    if (!e.target.closest('#menu-main-menu') && !e.target.closest('.hamburger-button')) {
      closeAllSubmenus();
    }
  });
});
