<?php

function create_team_post_type() {
    register_post_type('team',
        [
            'labels' => [
                'name' => __('Team Members'),
                'singular_name' => __('Team Member'),
                'add_new_item' => __('Add New Team Member'),
                'edit_item' => __('Edit Team Member'),
            ],
            'public' => true,
            'has_archive' => true,
            'show_in_rest' => true,
            'supports' => ['title', 'editor', 'thumbnail'],
            'menu_icon' => 'dashicons-groups',
        ]
    );
}
add_action('init', 'create_team_post_type');

function enqueue_team_member_admin_scripts($hook) {
    if ($hook !== 'post.php' && $hook !== 'post-new.php') return;

    wp_enqueue_media();
    wp_enqueue_script('team-member-meta-box', get_template_directory_uri() . '/js/team-member-meta-box.js', ['jquery'], null, true);
}
add_action('admin_enqueue_scripts', 'enqueue_team_member_admin_scripts');

function enqueue_team_member_assets() {
    wp_enqueue_style('team-member', get_template_directory_uri() . '/css/team-member.css');
}
add_action('wp_enqueue_scripts','enqueue_team_member_assets');

function add_team_member_meta_boxes() {
    // Image 1 Meta Box
    add_meta_box(
        'team_member_image_1',
        __('Team Member Front Image', 'xgate'),
        'team_member_image_callback',
        'team',
        'normal',
        'high',
        ['image_id' => 'team_member_image_1']
    );

    // Image 2 Meta Box
    add_meta_box(
        'team_member_image_2',
        __('Team Member Back Image', 'xgate'),
        'team_member_image_callback',
        'team',
        'normal',
        'high',
        ['image_id' => 'team_member_image_2']
    );

    // Image 3 Meta Box
    add_meta_box(
        'team_member_image_3',
        __('Team Member Image 3', 'xgate'),
        'team_member_image_callback',
        'team',
        'normal',
        'high',
        ['image_id' => 'team_member_image_3']
    );

    // Description Meta Box
    add_meta_box(
        'team_member_description',
        __('Team Member Description', 'xgate'),
        'team_member_description_callback',
        'team',
        'normal',
        'high'
    );

    // Role Meta Box
    add_meta_box(
        'team_member_role',
        __('Team Member Role', 'xgate'),
        'team_member_role_callback',
        'team',
        'side',
        'high'
    );

    // Role Meta Box
    add_meta_box(
        'team_member_gender',
        __('Team Member Gender', 'xgate'),
        'team_member_gender_callback',
        'team',
        'side',
        'high'
    );
}
add_action('add_meta_boxes', 'add_team_member_meta_boxes');

// Callback for Individual Image Meta Boxes
function team_member_image_callback($post, $meta_box) {
    $image_id = get_post_meta($post->ID, $meta_box['args']['image_id'], true);
    $image_src = $image_id ? wp_get_attachment_image_src($image_id, 'thumbnail')[0] : '';

    echo '<div class="team-member-image-box">';
    echo '<img width="50" src="' . esc_url($image_src) . '" style="max-width:100%; height:auto;" id="' . esc_attr($meta_box['args']['image_id'] . '_preview') . '" />';
    echo '<input type="hidden" name="' . esc_attr($meta_box['args']['image_id']) . '" id="' . esc_attr($meta_box['args']['image_id']) . '" value="' . esc_attr($image_id) . '" />';
    echo '<button type="button" class="button team-member-image-upload" data-target="' . esc_attr($meta_box['args']['image_id']) . '">' . __('Select Image', 'xgate') . '</button>';
    echo '</div>';
}

// Callback for Description Meta Box
function team_member_description_callback($post) {
    $description = get_post_meta($post->ID, 'team_member_description', true);
    echo '<textarea name="team_member_description" rows="5" style="width:100%;">' . esc_textarea($description) . '</textarea>';
}

// Callback for Gender Meta Box
function team_member_gender_callback($post) {
    $gender = get_post_meta($post->ID, 'team_member_gender', true);
    echo '<select name="team_member_gender">';
    echo '<option value="">' . __('Select a gender', 'xgate') . '</option>';
    echo '<option value="male" ' . selected($gender, 'male', false) . '>' . __('Male', 'xgate') . '</option>';
    echo '<option value="female" ' . selected($gender, 'female', false) . '>' . __('Female', 'xgate') . '</option>';
    echo '<option value="non_binary" ' . selected($gender, 'non_binary', false) . '>' . __('Non Binary', 'xgate') . '</option>';
    echo '</select>';
}

// Callback for Role Meta Box
function team_member_role_callback($post) {
    $role = get_post_meta($post->ID, 'team_member_role', true);
    $roles = [
        'CEO' => 'CEO',
        'partner' => 'Partner & Marketing Director',
        'UK' => 'UK Business Developer',
        'HR' => 'HR',
        'software_engineer' => 'Software Engineer',
        'qa_engineer' => 'QA Engineer',
        'graphics_web' => 'Graphic Designer / Web Designer',
        'seo_marketing' => 'SEO / Marketing',
        'seo' => 'SEO Specialist',
        'systems_admin' => 'Systems Admin',
        'engineer_manager' => 'Engineering Manager',
        'accounting' => 'Accounting',
        'office_manager' => 'Office Administrator',
        'graphisc_ssm' => 'Graphic Designer & SMM',
        'content_marketing' => 'Content / Marketing',
        'content_writer' => 'Content Writer',
    ];
    echo '<select name="team_member_role">';
    echo '<option value="">' . __('Select a role', 'xgate') . '</option>';
    foreach ($roles as $key => $label) {
        echo '<option value="' . esc_attr($key) . '" ' . selected($role, $key, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select>';
}

function save_team_member_meta($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    foreach (['team_member_image_1', 'team_member_image_2', 'team_member_image_3'] as $image_field) {
        if (isset($_POST[$image_field])) {
            update_post_meta($post_id, $image_field, sanitize_text_field($_POST[$image_field]));
        }
    }

    if (isset($_POST['team_member_description'])) {
        update_post_meta($post_id, 'team_member_description', sanitize_textarea_field($_POST['team_member_description']));
    }

    if (isset($_POST['team_member_role'])) {
        update_post_meta($post_id, 'team_member_role', sanitize_text_field($_POST['team_member_role']));
    }

    if (isset($_POST['team_member_gender'])) {
        update_post_meta($post_id, 'team_member_gender', sanitize_text_field($_POST['team_member_gender']));
    }
}

add_action('save_post', 'save_team_member_meta');





//add customize options 
function team_customizer_settings($wp_customize) {
    // Add a section for the Team page
    $wp_customize->add_section('team_section', array(
        'title'       => __('Team Page', 'xgate'),
        'description' => __('Customize the Team Page', 'xgate'),
        'priority'    => 30,
    ));

    // Add setting for the title
    $wp_customize->add_setting('team_title', array(
        'default'   => 'Default Team Title',
        'transport' => 'refresh',
    ));

    // Add control for the title
    $wp_customize->add_control(new WP_Customize_Control($wp_customize, 'team_title_control', array(
        'label'    => __('Team Title', 'xgate'),
        'section'  => 'team_section',
        'settings' => 'team_title',
        'type'     => 'text',
    )));

    // Add setting for the description
    $wp_customize->add_setting('team_description', array(
        'default'   => 'This is a default description for the Team page.',
        'transport' => 'refresh',
    ));

    // Add control for the description
    $wp_customize->add_control(new WP_Customize_Control($wp_customize, 'team_description_control', array(
        'label'    => __('Team Description', 'xgate'),
        'section'  => 'team_section',
        'settings' => 'team_description',
        'type'     => 'textarea',
    )));
}
add_action('customize_register', 'team_customizer_settings');