<?php

/**
 * Blocks Initializer
 *
 * Enqueue CSS/JS of all the blocks.
 *
 * @since   1.0.0
 * @package xgate
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue Gutenberg block assets for both frontend + backend.
 *
 * Assets enqueued:
 * 2. blocks.build.js - Backend.
 *
 * @uses {wp-blocks} for block type registration & related functions.
 * @uses {wp-element} for WP Element abstraction — structure of blocks.
 * @uses {wp-i18n} to internationalize the block's text.
 * @uses {wp-editor} for WP editor styles.
 * @since 1.0.0
 */
function xgate_stage_assets()
{ // phpcs:ignore

    // Register block editor script for backend.
    wp_register_script(
        'xgate-stage-js', // Handle.
        get_template_directory_uri() . '/inc/blocks/xgate-stage/build/xgate-stage.js', // Block.build.js: We register the block here. Built with Webpack.
        array('wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components'), // Dependencies, defined above.
        null, // filemtime( plugin_dir_path( __DIR__ ) . 'dist/blocks.build.js' ), // Version: filemtime — Gets file modification time.
        true // Enqueue the script in the footer.
    );

    // WP Localized globals. Use dynamic PHP stuff in JavaScript via `cgbGlobal` object.
    wp_localize_script(
        'xgate-stage-js',
        'cgbGlobal', // Array containing dynamic data for a JS Global.
        [
            'pluginDirPath' => plugin_dir_path(__DIR__),
            'pluginDirUrl' => plugin_dir_url(__DIR__),
        ]
    );

    /**
     * Register Gutenberg block on server-side.
     *
     * Register the block on server-side to ensure that the block
     * scripts and styles for both frontend and backend are
     * enqueued when the editor loads.
     *
     * @link https://wordpress.org/gutenberg/handbook/blocks/writing-your-first-block-type#enqueuing-block-scripts
     * @since 1.16.0
     */
    register_block_type(
        'xgate/stage',
        array(
            'editor_script' => 'xgate-stage-js',
            'render_callback' => 'stage_block_render'
        )
    );
}

function stage_block_render($attributes, $content)
{
    $title = isset($attributes['title']) ? $attributes['title'] : '';
    $subtitle = isset($attributes['subtitle']) ? $attributes['subtitle'] : '';
    $design = isset($attributes['design']) ? $attributes['design'] : 'blue-white';
    $imageUrl = isset($attributes['imageUrl']) ? $attributes['imageUrl'] : '';
    $content = isset($attributes['content']) ? $attributes['content'] : '';
    $variant = isset($attributes['variant']) ? $attributes['variant'] : 'content-center';

    // Add variant class to the block container
    $variant_class = 'my-custom-block ' . esc_attr($variant);
    // var_dump($design);
    $style = '';
    switch ($design) {
        case 'blue-white':
            $background_class = 'bg-xgate-dark-blue background-xgate-pattern';
            $style = 'style="background-image: url(\'' . get_template_directory_uri() . '/assets/images/background-xgate-lines-pattern.svg\');"';
            break;
        case 'white-blue':
            $background_class = 'background-xgate-digital-pattern bg-gray-50';
            $style = 'style="background-image: url(\'' . get_template_directory_uri() . '/assets/images/background-xgate-digital-pattern.svg\');"';
            break;
        case 'dark':
            $background_class = 'bg-xgate-dark-blue';
            break;
    }

    ob_start();
?>

    <section class="w-full overflow-hidden relative flex items-center <?php echo $background_class; ?>"
        <?php echo $style; ?>>
        <section class="w-full px-6 lg:px-16 mx-auto max-w-screen-2xl py-10 sm:py-20 flex flex-col lg:flex-row gap-x-24 gap-y-12  items-center justify-between lg:py-10 <?php echo $variant ?>">
            <div>
                <?php if ($design === 'white-blue') : ?>
                    <p class="h4-styles mb-5 text-black"><?php echo $subtitle; ?></p>
                    <h1 class="text-black mb-7 h1-big-styles font-medium"><?php echo $title; ?></h1>
                    <hr class="border-b-8 border-xgate-blue mb-5">
                    <div class="text-xl lg:text-2xl tracking-wide font-light max-w-2xl text-black">
                        <?php echo $content; ?>
                    </div>
                <?php else : ?>
                    <p class="h4-styles mb-5 text-white"><?php echo $subtitle; ?></p>
                    <h1 class="text-white mb-7"><?php echo $title; ?></h1>
                    <div class="text-xl lg:text-2xl tracking-wide font-light lg:max-w-2xl text-white">
                        <?php echo $content; ?>
                    </div>
                <?php endif; ?>
            </div>
            <!-- <img alt="xgate-hero-image" fetchpriority="high" decoding="async" data-nimg="fill"
                class="z-20 absolute xl:!min-w-[900px] !w-1/2 object-cover top-0 !right-0 !left-[unset] md:hidden lg:block overflow-x-visible" sizes="100vw" src="<?php echo $imageUrl; ?>"
                style="position: absolute; height: 100%; width: 100%; inset: 0px; color: transparent;"> -->
            <?php if ($imageUrl) : ?>
                <div class="relative w-full md:h-[250px] lg:h-[550px]">
                    <img alt="xgate-logo" fetchpriority="high" decoding="async" data-nimg="fill"
                        class="object-contain w-full lg:object-right" sizes="100vw" src="<?php echo $imageUrl; ?>"
                        style="position: absolute; height: 100%; width: 100%; inset: 0px; color: transparent;">
                </div>
            <?php endif; ?>
        </section>
    </section>

<?php
    return ob_get_clean();
}


// Hook: Block assets.
add_action('init', 'xgate_stage_assets');
