<?php

/**
 * Blocks Initializer
 *
 * Enqueue CSS/JS of all the blocks.
 *
 * @since   1.0.0
 * @package xgate
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue Gutenberg block assets for both frontend + backend.
 *
 * Assets enqueued:
 * 2. blocks.build.js - Backend.
 *
 * @uses {wp-blocks} for block type registration & related functions.
 * @uses {wp-element} for WP Element abstraction — structure of blocks.
 * @uses {wp-i18n} to internationalize the block's text.
 * @uses {wp-editor} for WP editor styles.
 * @since 1.0.0
 */
function xgate_solo_cta_assets()
{ // phpcs:ignore

    // Register block editor script for backend.
    wp_register_script(
        'xgate-solo-cta-js', // Handle.
        get_template_directory_uri() . '/inc/blocks/solo-cta/build/solo-cta.js', // Block.build.js: We register the block here. Built with Webpack.
        array('wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components'), // Dependencies, defined above.
        null, // filemtime( plugin_dir_path( __DIR__ ) . 'dist/blocks.build.js' ), // Version: filemtime — Gets file modification time.
        true // Enqueue the script in the footer.
    );

    // WP Localized globals. Use dynamic PHP stuff in JavaScript via `cgbGlobal` object.
    wp_localize_script(
        'xgate-solo-cta-js',
        'cgbGlobal', // Array containing dynamic data for a JS Global.
        [
            'pluginDirPath' => plugin_dir_path(__DIR__),
            'pluginDirUrl' => plugin_dir_url(__DIR__),
        ]
    );

    /**
     * Register Gutenberg block on server-side.
     *
     * Register the block on server-side to ensure that the block
     * scripts and styles for both frontend and backend are
     * enqueued when the editor loads.
     *
     * @link https://wordpress.org/gutenberg/handbook/blocks/writing-your-first-block-type#enqueuing-block-scripts
     * @since 1.16.0
     */
    register_block_type(
        'xgate/solo-cta',
        array(
            'editor_script' => 'xgate-solo-cta-js',
            'render_callback' => 'solo_cta_render'
        )
    );
}

function solo_cta_render($attributes, $content)
{
    $title = isset($attributes['title']) ? $attributes['title'] : '';
    $text = isset($attributes['textContent']) ? $attributes['textContent'] : '';
    $cta_link = isset($attributes['url']) ? $attributes['url'] : '';
    $cta_text = isset($attributes['text']) ? $attributes['text'] : '';
    $variant = isset($attributes['variant']) ? $attributes['variant'] : 'primary';

    $buttonText = isset($attributes['buttonText']) ? $attributes['buttonText'] : 'Click me';
    $design = isset($attributes['design']) ? $attributes['design'] : 'dark';

    $background_class = '';
    switch ($design) {
        case 'dark':
            $background_class = 'bg-xgate-dark-blue';
            break;
        case 'transparent':
            $background_class = 'transparent';
            break;
    }


    // Add variant class to the block container
    $variant_class = 'primary ' . esc_attr($variant);

    ob_start();
?>
    <section class="w-full py-20 relative <?php echo $background_class ?>">
        <div class="container mx-auto relative z-10">
            <section class="w-full px-6 lg:px-16 mx-auto max-w-screen-2xl undefined undefined">
                <div class="text-center flex flex-col py-14 gap-4 w-full ">
                    <div class="text-center flex flex-col gap-2 mx-auto">
                        <h2 class="w-full max-w-2xl mb-5 mx-auto "><?php echo $title ?></h2>
                        <p class="w-full leading-relaxed max-w-4xl mb-5"><?php echo $text ?></p>
                    </div>
                    <div class="flex flex-row gap-4 justify-center">
                        <a class="button <?php echo $variant ?>"
                            href="<?php echo $cta_link ?>" role="button">
                            <?php echo $buttonText ?>
                        </a>
                    </div>
                </div>
            </section>
        </div>
    </section>
<?php
    return ob_get_clean();
}


// Hook: Block assets.
add_action('init', 'xgate_solo_cta_assets');
