<?php
/**
 * Blocks Initializer
 *
 * Enqueue CSS/JS of all the blocks.
 *
 * @since   1.0.0
 * @package xgate
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue Gutenberg block assets for both frontend + backend.
 *
 * Assets enqueued:
 * 2. blocks.build.js - Backend.
 *
 * @uses {wp-blocks} for block type registration & related functions.
 * @uses {wp-element} for WP Element abstraction — structure of blocks.
 * @uses {wp-i18n} to internationalize the block's text.
 * @uses {wp-editor} for WP editor styles.
 * @since 1.0.0
 */
function xgate_quote_assets()
{ // phpcs:ignore

    // Register block editor script for backend.
    wp_register_script(
        'xgate-quote-js', // Handle.
        get_template_directory_uri() . '/inc/blocks/quote/build/quote.js', // Block.build.js: We register the block here. Built with Webpack.
        array( 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components'), // Dependencies, defined above.
        null, // filemtime( plugin_dir_path( __DIR__ ) . 'dist/blocks.build.js' ), // Version: filemtime — Gets file modification time.
        true // Enqueue the script in the footer.
    );

    // WP Localized globals. Use dynamic PHP stuff in JavaScript via `cgbGlobal` object.
    wp_localize_script(
        'xgate-quote-js',
        'cgbGlobal', // Array containing dynamic data for a JS Global.
        [
            'pluginDirPath' => plugin_dir_path(__DIR__),
            'pluginDirUrl' => plugin_dir_url(__DIR__),
        ]
    );

    /**
     * Register Gutenberg block on server-side.
     *
     * Register the block on server-side to ensure that the block
     * scripts and styles for both frontend and backend are
     * enqueued when the editor loads.
     *
     * @link https://wordpress.org/gutenberg/handbook/blocks/writing-your-first-block-type#enqueuing-block-scripts
     * @since 1.16.0
     */
    register_block_type(
        'xgate/quote', array(
            'editor_script' => 'xgate-quote-js',
            'render_callback' => 'quote_render'
        )
    );
}

function quote_render( $attributes, $content ) {
    $quote = isset( $attributes['quote'] ) ? $attributes['quote'] : '';
    $author = isset( $attributes['author'] ) ? $attributes['author'] : '';
    
    ob_start();
    ?>
<section class="py-5 sm:py-10">
    <div class="flex border-l-[20px] md:border-l-[40px] border-xgate-blue">
        <div class="w-full md:ml-[-20px]">
            <section class="w-full px-6 lg:px-16 mx-auto max-w-screen-2xl undefined undefined">
                <div class="flex gap-10 md:gap-36 flex-col md:flex-row">
                    <p class="max-w-5xl text-3xl  font-medium py-5 leading-[3rem]"><?php echo esc_html( $quote ); ?></p>
                    <div class="flex flex-col gap-5 py-5">
                        <p class="content-1 whitespace-nowrap font-medium"><?php echo esc_html( $author ); ?></p><svg
                            xmlns="http://www.w3.org/2000/svg" width="5rem" height="5rem" viewBox="0 0 24 24"
                            fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round"
                            stroke-linejoin="round" class="lucide lucide-quote text-xgate-blue">
                            <path
                                d="M16 3a2 2 0 0 0-2 2v6a2 2 0 0 0 2 2 1 1 0 0 1 1 1v1a2 2 0 0 1-2 2 1 1 0 0 0-1 1v2a1 1 0 0 0 1 1 6 6 0 0 0 6-6V5a2 2 0 0 0-2-2z">
                            </path>
                            <path
                                d="M5 3a2 2 0 0 0-2 2v6a2 2 0 0 0 2 2 1 1 0 0 1 1 1v1a2 2 0 0 1-2 2 1 1 0 0 0-1 1v2a1 1 0 0 0 1 1 6 6 0 0 0 6-6V5a2 2 0 0 0-2-2z">
                            </path>
                        </svg>
                    </div>
                </div>
            </section>
        </div>
    </div>
</section>
<?php
        return ob_get_clean();
    }
    

// Hook: Block assets.
add_action('init', 'xgate_quote_assets');