<?php

/**
 * Blocks Initializer
 *
 * Enqueue CSS/JS of all the blocks.
 *
 * @since   1.0.0
 * @package xgate
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue Gutenberg block assets for both frontend + backend.
 *
 * Assets enqueued:
 * 2. blocks.build.js - Backend.
 *
 * @uses {wp-blocks} for block type registration & related functions.
 * @uses {wp-element} for WP Element abstraction — structure of blocks.
 * @uses {wp-i18n} to internationalize the block's text.
 * @uses {wp-editor} for WP editor styles.
 * @since 1.0.0
 */
function xgate_image_block_assets()
{ // phpcs:ignore

    // Register block editor script for backend.
    wp_register_script(
        'xgate-image-block-js', // Handle.
        get_template_directory_uri() . '/inc/blocks/image-block/build/image-block.js', // Block.build.js: We register the block here. Built with Webpack.
        array('wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components'), // Dependencies, defined above.
        null, // filemtime( plugin_dir_path( __DIR__ ) . 'dist/blocks.build.js' ), // Version: filemtime — Gets file modification time.
        true // Enqueue the script in the footer.
    );

    // WP Localized globals. Use dynamic PHP stuff in JavaScript via `cgbGlobal` object.
    wp_localize_script(
        'xgate-image-block-js',
        'cgbGlobal', // Array containing dynamic data for a JS Global.
        [
            'pluginDirPath' => plugin_dir_path(__DIR__),
            'pluginDirUrl' => plugin_dir_url(__DIR__),
        ]
    );

    /**
     * Register Gutenberg block on server-side.
     *
     * Register the block on server-side to ensure that the block
     * scripts and styles for both frontend and backend are
     * enqueued when the editor loads.
     *
     * @link https://wordpress.org/gutenberg/handbook/blocks/writing-your-first-block-type#enqueuing-block-scripts
     * @since 1.16.0
     */
    register_block_type(
        'xgate/image-block',
        array(
            'editor_script' => 'xgate-image-block-js',
            'render_callback' => 'image_block_render'
        )
    );
}

function image_block_render($attributes, $content)
{
    $title = isset($attributes['title']) ? $attributes['title'] : '';
    $imageUrl = isset($attributes['imageUrl']) ? $attributes['imageUrl'] : '';
    $variant = isset($attributes['variant']) ? $attributes['variant'] : 'content-center';
    $design = isset($attributes['design']) ? $attributes['design'] : 'no-effect';
    $imageSize = isset($attributes['imageSize']) ? $attributes['imageSize'] : 'auto-height';

    // Add variant class to the block container
    $variant_class = 'image-block ' . esc_attr($variant);

    ob_start();
?>
    <div class="rounded-md xgate-image-block relative <?php echo $variant_class; ?> <?php echo $design; ?> <?php echo $imageSize; ?>">
        <img src="<?php echo esc_url($imageUrl); ?>" alt="<?php echo esc_attr($title); ?>" class='h-full w-full object-contain'/>
    </div>
<?php
    return ob_get_clean();
}


// Hook: Block assets.
add_action('init', 'xgate_image_block_assets');
