<?php

/**
 * Blocks Initializer
 *
 * Enqueue CSS/JS of all the blocks.
 *
 * @since   1.0.0
 * @package xgate
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue Gutenberg block assets for both frontend + backend.
 *
 * Assets enqueued:
 * 2. blocks.build.js - Backend.
 *
 * @uses {wp-blocks} for block type registration & related functions.
 * @uses {wp-element} for WP Element abstraction — structure of blocks.
 * @uses {wp-i18n} to internationalize the block's text.
 * @uses {wp-editor} for WP editor styles.
 * @since 1.0.0
 */
function xgate_content_cta_assets()
{ // phpcs:ignore

    // Register block editor script for backend.
    wp_register_script(
        'xgate-content-cta-js', // Handle.
        get_template_directory_uri() . '/inc/blocks/content-cta/build/content-cta.js', // Block.build.js: We register the block here. Built with Webpack.
        array('wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components'), // Dependencies, defined above.
        null, // filemtime( plugin_dir_path( __DIR__ ) . 'dist/blocks.build.js' ), // Version: filemtime — Gets file modification time.
        true // Enqueue the script in the footer.
    );

    // WP Localized globals. Use dynamic PHP stuff in JavaScript via `cgbGlobal` object.
    wp_localize_script(
        'xgate-content-cta-js',
        'cgbGlobal', // Array containing dynamic data for a JS Global.
        [
            'pluginDirPath' => plugin_dir_path(__DIR__),
            'pluginDirUrl' => plugin_dir_url(__DIR__),
        ]
    );

    /**
     * Register Gutenberg block on server-side.
     *
     * Register the block on server-side to ensure that the block
     * scripts and styles for both frontend and backend are
     * enqueued when the editor loads.
     *
     * @link https://wordpress.org/gutenberg/handbook/blocks/writing-your-first-block-type#enqueuing-block-scripts
     * @since 1.16.0
     */
    register_block_type(
        'xgate/content-cta',
        array(
            'editor_script' => 'xgate-content-cta-js',
            'render_callback' => 'content_cta_render'
        )
    );
}

function content_cta_render($attributes, $content)
{
    $title = isset($attributes['title']) ? $attributes['title'] : '';
    $subtitle = isset($attributes['subtitle']) ? $attributes['subtitle'] : '';
    $text = isset($attributes['textContent']) ? $attributes['textContent'] : '';
    $cta_link = isset($attributes['url']) ? $attributes['url'] : '';
    $cta_text = isset($attributes['text']) ? $attributes['text'] : '';
    $variant = isset($attributes['variant']) ? $attributes['variant'] : 'primary';
    $buttonText = isset($attributes['buttonText']) ? $attributes['buttonText'] : 'Click me';

    // Add variant class to the block container
    $variant_class = 'primary ' . esc_attr($variant);

    ob_start();
?>
    <div class="flex gap-4 flex-col content-cta">
        <?php if ($subtitle): ?>
            <div class="content-cta__subtitle relative max-w-fit min-w-min inline-flex items-center justify-between box-border whitespace-nowrap border-medium px-1 h-7 text-small rounded-full bg-transparent">
                <span class="flex-1 text-inherit font-normal px-2"><?php echo $subtitle; ?></span>
            </div>
        <?php endif; ?>
        <div class="flex gap-4 flex-col">
            <h2 class="max-w-lg md:text-left sm:text-center font-medium"><?php echo $title ?></h2>
            <div class="content-1 leading-relaxed sm:text-center md:text-left"><?php echo $text ?></div>
        </div>
            <div class="flex flex-row gap-4">
                <div class="sm:mt-5 md:mt-8 sm:text-center md:text-left">
                    <a class="button <?php echo $variant ?> content-cta__button" role="button" tabindex="0"
                        href="<?php echo $cta_link ?>">
                        <?php echo $buttonText ?>
               
                    </a>
                </div>
            </div>
    </div>
<?php
    return ob_get_clean();
}


// Hook: Block assets.
add_action('init', 'xgate_content_cta_assets');
