<?php

namespace Drupal\formstack\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigManager;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Site\Settings;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class FormstackConfigForm.
 *
 * Provides a configuration form for Formstack integration.
 */
class FormstackConfigForm extends ConfigFormBase {

  /**
   * The configuration manager service.
   *
   * @var \Drupal\Core\Config\ConfigManager
   */
  protected $config_manager;

  /**
   * The messenger service for displaying messages.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a FormstackConfigForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed configuration manager.
   * @param \Drupal\Core\Config\ConfigManager $config_manager
   *   The configuration manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    ConfigManager $config_manager,
    MessengerInterface $messenger
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->config_manager = $config_manager;
    $this->messenger = $messenger;
  }

  /**
   * Creates an instance of the form using the service container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The instantiated form class.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('config.manager'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['formstack.config'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'formstack_config_form';
  }

  /**
   * Builds the configuration form.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array
   *   The built form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('formstack.config');

    $form['step_one'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Step 1: Register your application'),
    ];
    $form['step_one']['instructions'] = [
      '#type' => 'markup',
      '#markup' => $this->t($this->formstack_application_instructions_step_one_markup()),
    ];

    $form['step_two'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Step 2: Authorization'),
    ];
    $form['step_two']['instructions'] = [
      '#type' => 'markup',
      '#markup' => $this->t($this->formstack_application_instructions_step_two_markup()),
    ];
    $form['step_two']['access_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Token'),
      '#maxlength' => 64,
      '#size' => 64,
      '#default_value' => $config->get('access_token'),
    ];
    $form['step_two']['access_token_instructions'] = [
      '#type' => 'markup',
      '#markup' => $this->t('You can either set the access token here or by adding `$settings["formstack_access_token"];` to your settings.php file.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * Generates markup instructions for Step 1.
   *
   * @return string
   *   The HTML markup for the instructions.
   */
  public function formstack_application_instructions_step_one_markup() {
    global $base_url;
    $site_name = \Drupal::state()->get('site_name');
    $redirect_url = $base_url . $this->current_path();

    $output = <<<FAP
      <h2>You must first register a new application in your Formstack account.</h2>
      <p>Login to your Formstack account and proceed to <a href="https://www.formstack.com/admin/applications/add" target="_blank">https://www.formstack.com/admin/applications/add</a></p>
      <p>In the fields enter the following values:</p>
      <ul>
        <li><strong>Application Name:</strong> $site_name Website</li>
        <li><strong>Redirect URI:</strong> $redirect_url</li>
        <li><strong>Description:</strong> <em>(anything you want)</em></li>
        <li><strong>Logo:</strong> <em>(optional)</em></li>
        <li><strong>Platform:</strong> Website</li>
      </ul>
      <p>Click "Create Application" button</p>
    FAP;

    return $output;
  }

  /**
   * Generates markup instructions for Step 2.
   *
   * @return string
   *   The HTML markup for the instructions.
   */
  public function formstack_application_instructions_step_two_markup() {
    global $base_url;
    $site_name = \Drupal::state()->get('site_name');
    $redirect_url = $base_url . $this->current_path();

    $output = <<<FAP
      <h2>Enter your Access Token below from the application you created in Step 1.</h2>
      <p>From the <a href="https://www.formstack.com/admin/apiKey/main" target="_blank">Applications page on Formstack</a>, click on the application icon. You will find the details in a box on the right side of the window Access Token *</p>
    FAP;

    return $output;
  }

  /**
   * Gets the current path.
   *
   * @return string
   *   The current path.
   */
  public function current_path() {
    return \Drupal::service('path.current')->getPath();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    if (Settings::get('formstack_access_token') !== NULL) {
      $this->messenger->addWarning('The access token is currently overridden in the settings file.');
      return;
    }

    $this->config('formstack.config')
      ->set('access_token', $form_state->getValue('access_token'))
      ->save();
  }
}
