<?php

namespace Drupal\dna_migration\Plugin\migrate\source;

use Drupal\migrate\Plugin\migrate\source\SourcePluginBase;
use Drupal\migrate\Plugin\MigrationInterface;

/**
 * Source plugin for entityform submissions from XML.
 *
 * @MigrateSource(
 *   id = "entityform_source"
 * )
 */
class EntityFormSource extends SourcePluginBase {

  /**
   * The file path to the XML.
   *
   * @var string
   */
  protected $filePath;

  /**
   * The parsed XML document.
   *
   * @var \SimpleXMLElement
   */
  protected $xml;

  /**
   * The submissions array.
   *
   * @var array
   */
  protected $submissions = [];

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $migration);
    $this->filePath = $configuration['file_path'];
  }

  /**
   * {@inheritdoc}
   */
  public function initializeIterator() {
    $this->loadXml();
    return new \ArrayIterator($this->submissions);
  }

  /**
   * Load and parse the XML file.
   */
  protected function loadXml() {
    $fileSystem = \Drupal::service('file_system');
    $realPath = $fileSystem->realpath($this->filePath);

    if (!$realPath || !file_exists($realPath) || !is_readable($realPath)) {
      \Drupal::logger('dna_migration')->error('XML file not found or not readable: @path', ['@path' => $this->filePath]);
      return;
    }

    try {
      $xmlContent = file_get_contents($realPath);
      $this->xml = simplexml_load_string($xmlContent);

      if ($this->xml === false) {
        $errors = libxml_get_errors();
        $errorMessages = [];
        foreach ($errors as $error) {
          $errorMessages[] = $error->message;
        }
        \Drupal::logger('dna_migration')->error('Failed to parse XML file: @path. Errors: @errors', [
          '@path' => $this->filePath,
          '@errors' => implode('; ', $errorMessages),
        ]);
        return;
      }

      $index = 0;
      foreach ($this->xml->entityform as $entityform) {
        $submission = [];
        foreach ($entityform->children() as $field) {
          $fieldName = (string) $field->getName();
          $fieldValue = (string) $field;

          // Clean Email field.
          if ($fieldName == 'Email') {
            if (preg_match('/mailto:([^"]+)/', $fieldValue, $matches)) {
              $fieldValue = $matches[1];
            } else {
              $fieldValue = strip_tags($fieldValue);
            }
          }

          $submission[$fieldName] = $fieldValue;
        }

        // Force Submitter to Anonymous.
        $submission['Submitter'] = 'Anonymous';

        // Set default values for GA fields.
        $submission['GA-Campaign-Content'] = $submission['GA-Campaign-Content'] ?? '';
        $submission['GA-Campaign-Medium'] = $submission['GA-Campaign-Medium'] ?? '';
        $submission['GA-Campaign-Name'] = $submission['GA-Campaign-Name'] ?? '';
        $submission['GA-Campaign-Source'] = $submission['GA-Campaign-Source'] ?? '';
        $submission['GA-Campaign-Term'] = $submission['GA-Campaign-Term'] ?? '';
        $submission['GA-Cid'] = $submission['GA-User-Data'] ?? '';

        // Generate unique_id (fallback to index if Date-submitted or Email is empty).
        $submission['unique_id'] = !empty($submission['Date-submitted']) && !empty($submission['Email'])
          ? $submission['Date-submitted'] . '|' . $submission['Email']
          : 'fallback_' . $index;

        $this->submissions[] = $submission;
        $index++;
      }

      \Drupal::logger('dna_migration')->notice('Successfully loaded @count entityform submissions.', [
        '@count' => count($this->submissions),
      ]);
    }
    catch (\Exception $e) {
      \Drupal::logger('dna_migration')->error('Error loading XML file: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function fields() {
    return [
      'unique_id' => $this->t('Unique Identifier'),
      'Submitter' => $this->t('Submitter'),
      'Form-Type' => $this->t('Form Type'),
      'Date-submitted' => $this->t('Date submitted'),
      'Date-changed' => $this->t('Date changed'),
      'First-name' => $this->t('First name'),
      'Last-name' => $this->t('Last name'),
      'Email' => $this->t('Email'),
      'Phone' => $this->t('Phone'),
      'Testing---Enquiry-type' => $this->t('Enquiry type'),
      'Details---Questions' => $this->t('Details/Questions'),
      'Company' => $this->t('Company'),
      'Referral-Page' => $this->t('Referral Page'),
      'GA-Campaign-Content' => $this->t('GA Campaign Content'),
      'GA-Campaign-Medium' => $this->t('GA Campaign Medium'),
      'GA-Campaign-Name' => $this->t('GA Campaign Name'),
      'GA-Campaign-Source' => $this->t('GA Campaign Source'),
      'GA-Campaign-Term' => $this->t('GA Campaign Term'),
      'GA-Cid' => $this->t('GA CID'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getIds() {
    return [
      'unique_id' => [
        'type' => 'string',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function count($refresh = false): int {
    if ($refresh || empty($this->submissions)) {
      $this->loadXml();
    }
    return count($this->submissions);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString() {
    return 'EntityformSource from file: ' . $this->filePath;
  }
}
