<?php

namespace Drupal\dna_migration\Plugin\migrate\process;

use Drupal\migrate\MigrateExecutableInterface;
use Drupal\migrate\ProcessPluginBase;
use Drupal\migrate\Row;
use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Formats a date field properly for migration.
 *
 * @MigrateProcessPlugin(
 *   id = "date_format"
 * )
 */
class DateFormatter extends ProcessPluginBase {

  /**
   * Transforms the given date into a formatted string.
   *
   * @param mixed $value
   *   The date value from the source data.
   * @param \Drupal\migrate\MigrateExecutableInterface $migrate_executable
   *   The migrate executable.
   * @param \Drupal\migrate\Row $row
   *   The row object.
   * @param string $destination_property
   *   The destination field property.
   *
   * @return string|null
   *   The formatted date string in ISO 8601 format or NULL if empty.
   */
  public function transform($value, MigrateExecutableInterface $migrate_executable, Row $row, $destination_property) {
    if (empty($value)) {
      return NULL;
    }


    try {
      $date = new DrupalDateTime($value);
      // Format the date to ISO 8601 format (e.g., 2025-03-18T03:03:56)
      return $date->format('Y-m-d\TH:i:s');
    }
    catch (\Exception $e) {
      \Drupal::logger('dna_migration')->error('Invalid date format: @value', ['@value' => $value]);
      return NULL;
    }
  }
}
