<?php

namespace Drupal\dna_migration\EventSubscriber;

use Drupal\migrate\Event\MigrateEvents;
use Drupal\migrate\Event\MigratePostRowSaveEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\user\Entity\User;

/**
 * Event subscriber to map and assign roles after migration.
 */
class PostMigrationRoleMapper implements EventSubscriberInterface {

  /**
   * Mapping of Drupal 7 role IDs (rid) to Drupal 10 role machine names.
   *
   * @var array
   */
  protected $roleMap = [
    3 => 'administrator',
    4 => 'content_editor',
    5 => 'portal_user_admin',
    6 => 'portal_user',
    7 => NULL,
  ];

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      MigrateEvents::POST_ROW_SAVE => 'onPostRowSave',
    ];
  }

  /**
   * Handle the post row save event to assign roles.
   *
   * @param \Drupal\migrate\Event\MigratePostRowSaveEvent $event
   *   The migration post row save event.
   */
  public function onPostRowSave(MigratePostRowSaveEvent $event) {
    $migration = $event->getMigration();
    if ($migration->id() !== 'dna_users') {
      return;
    }

    $row = $event->getRow();
    $destination_id_values = $event->getDestinationIdValues();
    $uid = reset($destination_id_values); // Get the migrated user ID.

    // Load the user entity.
    $user = User::load($uid);
    if (!$user) {
      \Drupal::logger('dna_migration')->error('Failed to load user with UID: @uid', ['@uid' => $uid]);
      return;
    }

    // Get Drupal 7 role IDs from the source row.
    $source_roles = $row->getSourceProperty('roles') ?? [];
    if (empty($source_roles)) {
      \Drupal::logger('dna_migration')->debug('No source roles found for user: @uid', ['@uid' => $uid]);
      return;
    }

    // Map Drupal 7 rids to Drupal 10 machine names.
    $mapped_roles = [];
    foreach ($source_roles as $rid) {
      if (is_numeric($rid) && isset($this->roleMap[$rid]) && $this->roleMap[$rid] !== NULL) {
        $mapped_roles[] = $this->roleMap[$rid];
      } else {
        \Drupal::logger('dna_migration')->warning('No Drupal 10 role mapping found for Drupal 7 rid: @rid for user @uid', [
          '@rid' => $rid,
          '@uid' => $uid,
        ]);
      }
    }

    if (empty($mapped_roles)) {
      \Drupal::logger('dna_migration')->debug('No valid roles mapped for user: @uid', ['@uid' => $uid]);
      return;
    }

    foreach ($mapped_roles as $role_id) {
      $user->addRole($role_id);
    }

    // Save the user with updated roles.
    $user->save();

    \Drupal::logger('dna_migration')->debug('Assigned roles to user @uid: @roles', [
      '@uid' => $uid,
      '@roles' => implode(', ', $mapped_roles),
    ]);
  }
}
