#!/bin/bash

# Check if running inside a DDEV container
if [ -n "$DDEV_PROJECT" ]; then
  echo "Error: This script must run on the host, not inside a DDEV container."
  echo "Run './export_entityforms.sh' from '~/projects/external/dna_legal', not with 'ddev exec'."
  exit 1
fi

# Verify DDEV project directory
if ! ddev describe >/dev/null 2>&1; then
  echo "Error: Must be run inside a DDEV project directory."
  exit 1
fi

# Configuration
DB_NAME="dna_legal_old"
DB_USER="root"
DB_PASS="root"
OUTPUT_DIR="entityform_chunks"
FINAL_XML="web/sites/default/files/migrations/contact_us_form_data.xml"
CHUNK_SIZE=1000

mkdir -p "$OUTPUT_DIR"
mkdir -p "$(dirname "$FINAL_XML")"

# Check DB access
echo "Checking database access..."
if ! ddev mysql -u"$DB_USER" -p"$DB_PASS" -e "USE $DB_NAME;" >/dev/null 2>&1; then
  echo "Error: Unable to access database '$DB_NAME'."
  exit 1
fi

# Get ID range for contact_form
echo "Getting ID range..."
read MIN_ID MAX_ID <<<$(ddev mysql -u"$DB_USER" -p"$DB_PASS" -N -e \
  "SELECT MIN(entityform_id), MAX(entityform_id) FROM $DB_NAME.entityform WHERE LOWER(type) = 'contact_form';")

if [ -z "$MIN_ID" ] || [ -z "$MAX_ID" ]; then
  echo "No records found for type 'contact_form'."
  exit 0
fi

# Clean previous chunks
rm -f "$OUTPUT_DIR"/chunk_*.xml

# Export loop
CURRENT_ID=$MIN_ID
CHUNK_NUM=0
echo -n "Exporting"

while [ $CURRENT_ID -le $MAX_ID ]; do
  END_ID=$((CURRENT_ID + CHUNK_SIZE - 1))
  [ $END_ID -gt $MAX_ID ] && END_ID=$MAX_ID

  CHUNK_FILE="$OUTPUT_DIR/chunk_$CHUNK_NUM.xml"
  echo "Chunk $CHUNK_NUM: $CURRENT_ID to $END_ID"

  QUERY=$(cat <<EOF
SELECT CONCAT(
  '<entityform>\n',
  '<Submitter>\n    ',
    IF(ef.uid = 0,
      '<span class="username" xml:lang="" typeof="sioc:UserAccount" property="foaf:name" datatype="">Anonymous</span>',
      CONCAT('<a href="/users/', LOWER(REPLACE(u.name, ' ', '-')), '" title="View user profile." class="username" typeof="sioc:UserAccount" property="foaf:name">', u.name, '</a>')
    ),
  '\n</Submitter>\n',
  '<Form-Type>Request More Information</Form-Type>\n',
  '<Date-submitted>',
    DATE_FORMAT(FROM_UNIXTIME(ef.created), '%a, %m/%d/%Y - %H:%i'),
  '</Date-submitted>\n',
  '<Date-changed>',
    DATE_FORMAT(FROM_UNIXTIME(ef.changed), '%a, %m/%d/%Y - %H:%i'),
  '</Date-changed>\n',
  '<First-name>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_forename.field_forename_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</First-name>\n',
  '<Last-name>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_surname.field_surname_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</Last-name>\n',
  '<Email><a href="mailto:',
    REPLACE(REPLACE(REPLACE(IFNULL(field_email.field_email_email, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '">',
    REPLACE(REPLACE(REPLACE(IFNULL(field_email.field_email_email, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</a></Email>\n',
  '<Phone>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_phone.field_phone_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</Phone>\n',
  '<Testing---Enquiry-type>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_enquiry.field_contact_enquiry_type_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</Testing---Enquiry-type>\n',
  '<Details---Questions>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_desc.field_description_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</Details---Questions>\n',
  '<Company>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_company.field_company_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</Company>\n',
  '<GA-User-Data></GA-User-Data>\n',
  '<Referral-Page>',
    REPLACE(REPLACE(REPLACE(IFNULL(field_referral.field_referral_page_value, ''), '&', '&amp;'), '<', '&lt;'), '>', '&gt;'),
  '</Referral-Page>\n',
  '</entityform>\n'
) AS xml_output
FROM $DB_NAME.entityform ef
LEFT JOIN $DB_NAME.users u ON ef.uid = u.uid
LEFT JOIN $DB_NAME.field_data_field_forename field_forename ON ef.entityform_id = field_forename.entity_id
LEFT JOIN $DB_NAME.field_data_field_surname field_surname ON ef.entityform_id = field_surname.entity_id
LEFT JOIN $DB_NAME.field_data_field_email field_email ON ef.entityform_id = field_email.entity_id
LEFT JOIN $DB_NAME.field_data_field_phone field_phone ON ef.entityform_id = field_phone.entity_id
LEFT JOIN $DB_NAME.field_data_field_contact_enquiry_type field_enquiry ON ef.entityform_id = field_enquiry.entity_id
LEFT JOIN $DB_NAME.field_data_field_description field_desc ON ef.entityform_id = field_desc.entity_id
LEFT JOIN $DB_NAME.field_data_field_company field_company ON ef.entityform_id = field_company.entity_id
LEFT JOIN $DB_NAME.field_data_field_referral_page field_referral ON ef.entityform_id = field_referral.entity_id
WHERE LOWER(ef.type) = 'contact_form'
AND ef.entityform_id BETWEEN $CURRENT_ID AND $END_ID
ORDER BY ef.created;
EOF
)

  ddev mysql -u"$DB_USER" -p"$DB_PASS" -N -e "$QUERY" | grep -v '^NULL$' > "$CHUNK_FILE"

  if [ $? -ne 0 ]; then
    echo "Error on chunk $CHUNK_NUM"
    exit 1
  fi

  CURRENT_ID=$((END_ID + 1))
  CHUNK_NUM=$((CHUNK_NUM + 1))
  echo -n "."
done

# Combine chunks
echo -e "\nCombining chunks into $FINAL_XML..."
{
  echo '<?xml version="1.0" encoding="UTF-8" ?>'
  echo '<entityforms>'
  cat "$OUTPUT_DIR"/chunk_*.xml
  echo '</entityforms>'
} > "$FINAL_XML"

# Cleanup
rm -rf "$OUTPUT_DIR"

# Done
COUNT=$(grep -o "<entityform>" "$FINAL_XML" | wc -l)
echo "✅ Export complete: $FINAL_XML"
echo "📦 Total records exported: $COUNT"
