const gulp = require('gulp');
const sass = require('gulp-sass')(require('sass'));
const postcss = require('gulp-postcss');
const autoprefixer = require('autoprefixer');
const sourcemaps = require('gulp-sourcemaps');
const browserSync = require('browser-sync').create();
const rename = require('gulp-rename');
const cssnano = require('gulp-cssnano');
const fs = require('fs');
const path = require('path');
const { glob } = require('glob');
const uglifyEs = require('gulp-uglify-es').default;

// Component paths
const paths = {
  components: {
    scss: 'components/**/src/**/*.scss',
    js: 'components/**/src/**/*.js'
  },
  theme: {
    scss: 'src/scss/**/*.scss',
    js: 'src/js/**/*.js'
  },
  dist: {
    base: './',
    css: 'dist/css',
    js: 'dist/js'
  },
  watch: {
    scss: ['components/**/src/**/*.scss', 'src/scss/**/*.scss'],
    js: ['components/**/src/**/*.js', 'src/js/**/*.js'],
    twig: '**/*.twig'
  }
};

// Check if directory exists
function directoryExists(dirPath) {
  try {
    return fs.statSync(dirPath).isDirectory();
  } catch (err) {
    return false;
  }
}

// Clean files
async function clean(done) {
  try {
    const files = await glob(['**/*.min.css', '**/*.min.js', '**/*.map']);
    for (const file of files) {
      try {
        fs.unlinkSync(file);
      } catch (err) {
        // Ignore errors if file doesn't exist
      }
    }
    console.log('Clean completed');
    done();
  } catch (err) {
    console.error('Clean error:', err);
    done();
  }
}

// Compile Component SCSS for development
function stylesComponentsDev() {
  if (!glob.sync(paths.components.scss).length) {
    console.log('No component SCSS files found to process');
    return Promise.resolve('No component SCSS files found');
  }

  return gulp.src(paths.components.scss, { base: '.' })
  .pipe(sourcemaps.init())
  .pipe(sass({ outputStyle: 'expanded' }).on('error', sass.logError))
  .pipe(postcss([autoprefixer()]))
  .pipe(rename(function(path) {
    path.dirname = path.dirname.replace(/\/src$|\/src\//, '');
  }))
  .pipe(gulp.dest('./'))
  .pipe(sourcemaps.write('.'))
  .pipe(browserSync.stream({match: '**/*.css'}));
}

// Compile Theme SCSS for development
function stylesThemeDev() {
  if (!glob.sync(paths.theme.scss).length) {
    console.log('No theme SCSS files found to process');
    return Promise.resolve('No theme SCSS files found');
  }

  return gulp.src(paths.theme.scss)
  .pipe(sourcemaps.init())
  .pipe(sass({ outputStyle: 'expanded' }).on('error', sass.logError))
  .pipe(postcss([autoprefixer()]))
  .pipe(gulp.dest(paths.dist.css))
  .pipe(sourcemaps.write('.'))
  .pipe(browserSync.stream({match: '**/*.css'}));
}

// Compile Component SCSS for production
function stylesComponentsProd() {
  if (!glob.sync(paths.components.scss).length) {
    console.log('No component SCSS files found to process');
    return Promise.resolve('No component SCSS files found');
  }

  return gulp.src(paths.components.scss, { base: '.' })
  .pipe(sass({ outputStyle: 'compressed' }).on('error', sass.logError))
  .pipe(postcss([autoprefixer()]))
  .pipe(rename(function(path) {
    path.dirname = path.dirname.replace(/\/src$|\/src\//, '');
  }))
  .pipe(gulp.dest('./'))
  .pipe(cssnano())
  .pipe(rename({ suffix: '.min' }))
  .pipe(gulp.dest('./'));
}

// Compile Theme SCSS for production
function stylesThemeProd() {
  if (!glob.sync(paths.theme.scss).length) {
    console.log('No theme SCSS files found to process');
    return Promise.resolve('No theme SCSS files found');
  }

  return gulp.src(paths.theme.scss)
  .pipe(sass({ outputStyle: 'compressed' }).on('error', sass.logError))
  .pipe(postcss([autoprefixer()]))
  .pipe(gulp.dest(paths.dist.css))
  .pipe(cssnano())
  .pipe(rename({ suffix: '.min' }))
  .pipe(gulp.dest(paths.dist.css));
}

// Compile component JS dev
function scriptsComponentsDev() {
  if (!glob.sync(paths.components.js).length) {
    console.log('No component JS files found to process');
    return Promise.resolve('No component JS files found');
  }

  return gulp.src(paths.components.js, { base: '.' })
  .pipe(sourcemaps.init())
  .pipe(rename(function(path) {
    path.dirname = path.dirname.replace(/\/src$|\/src\//, '');
  }))
  .pipe(gulp.dest('./'))
  .pipe(sourcemaps.write('.'));
}

// Compile JS dev
function scriptsThemeDev() {
  if (!glob.sync(paths.theme.js).length) {
    console.log('No theme JS files found to process');
    return Promise.resolve('No theme JS files found');
  }

  return gulp.src(paths.theme.js)
  .pipe(sourcemaps.init())
  .pipe(gulp.dest(paths.dist.js))
  .pipe(sourcemaps.write('.'));
}

// Prod JS from components
function scriptsComponentsProd() {
  if (!glob.sync(paths.components.js).length) {
    console.log('No component JS files found to process');
    return Promise.resolve('No component JS files found');
  }

  return gulp.src(paths.components.js, { base: '.' })
  .pipe(rename(function(path) {
    path.dirname = path.dirname.replace(/\/src$|\/src\//, '');
  }))
  .pipe(gulp.dest('./'))
  .pipe(uglifyEs().on('error', function(err) {
    console.error('UglifyEs error:', err.toString());
    this.emit('end');
  }))
  .pipe(rename({ suffix: '.min' }))
  .pipe(gulp.dest('./'));
}

// Prod JS from theme
function scriptsThemeProd() {
  if (!glob.sync(paths.theme.js).length) {
    console.log('No theme JS files found to process');
    return Promise.resolve('No theme JS files found');
  }

  return gulp.src(paths.theme.js)
  .pipe(gulp.dest(paths.dist.js))
  .pipe(uglifyEs().on('error', function(err) {
    console.error('UglifyEs error:', err.toString());
    this.emit('end');
  }))
  .pipe(rename({ suffix: '.min' }))
  .pipe(gulp.dest(paths.dist.js));
}

// Reload browser after JS changes
function reloadBrowser(done) {
  browserSync.reload();
  done();
}

function watch() {
  browserSync.init({
    proxy: 'http://dna-legal.ddev.site',
    open: false,
    logFileChanges: false,
    logLevel: "info"
  });

  if (directoryExists('components')) {
    gulp.watch('components/**/src/**/*.scss', stylesComponentsDev);
    gulp.watch('components/**/src/**/*.js', gulp.series(scriptsComponentsDev, reloadBrowser));
  }

  if (directoryExists('src/scss')) {
    gulp.watch('src/scss/**/*.scss', stylesThemeDev);
  }

  if (directoryExists('src/js')) {
    gulp.watch('src/js/**/*.js', gulp.series(scriptsThemeDev, reloadBrowser));
  }

  gulp.watch(paths.watch.twig).on('change', browserSync.reload);
}

function stylesDev(done) {
  gulp.parallel(stylesComponentsDev, stylesThemeDev)(done);
}

function scriptsDev(done) {
  gulp.parallel(scriptsComponentsDev, scriptsThemeDev)(done);
}

const dev = gulp.series(
  gulp.parallel(stylesDev, scriptsDev),
  watch
);

const build = gulp.series(
  clean,
  gulp.parallel(
    stylesComponentsProd,
    stylesThemeProd,
    scriptsComponentsProd,
    scriptsThemeProd
  )
);

// Export tasks
exports.clean = clean;
exports.stylesDev = stylesDev;
exports.scriptsDev = scriptsDev;
exports.watch = watch;
exports.dev = dev;
exports.build = build;

// Default task
exports.default = dev;
